\version "2.22.2"

\header {
  texidoc = "
Sometimes you may want to get a sorted index of the pieces contained in
a book. Often the @code{\\table-of-content} command does this job quite
well. However, the problem with it is that it always sorts by page
numbers. For books containing many songs it is common to have an index
sorted by letters or topics.


Both is possible with this snippet. It takes the code of LilyPond's
@code{ly/toc-init.ly} file and defines three new commands
@code{\\indexItem}, @code{\\indexSection}, and @code{\\index}. The
first two are used to define index items, the last one will create the
output.


@code{\\indexItem} and @code{\\indexSection} both take two arguments
(string, markup). The first is used to calculate the order of the items
in the output, the second one contains the markup that is actually
printed.



  \\markuplist \\index

  \\indexItem #\"Berta\" \\markup \"Berta\"
  \\indexItem #\"Clown\" \\markup \"Clown\"
  \\indexItem #\"Adam\" \\markup \"Adam\"


By setting the first argument of @code{\\indexItem} and
@code{\\indexSection} to a different value than the second it is
possible to create heavily customized sorted indices. For example, a
topic-based index can be achieved by



  \\markuplist \\index

  \\indexSection #\"spring\" \\markup \\smallCaps \"Songs of Spring\"
  \\indexItem #\"spring - I Like the Flowers\" \\markup \"I Like the
Flowers\"
  \\indexItem #\"spring - Der Winter ist vergangen\" \\markup \"Der
Winter ist vergangen\"

  \\indexSection #\"rock\" \\markup \\smallCaps \"Rock Songs\"
  \\indexItem #\"rock - Stairway to Heaven\" \\markup \"Stairway to
heaven\"
  \\indexItem #\"rock - Nothing Else Matters\" \\markup \"Nothing Else
Matters\"


Limitations:


It is not yet possible to create multiple indices, for example, an
alphabetically sorted and a topic-based one. To achieve that a few
additions to the snippet would be necessary, which I had not yet time
to implement. 

"
  doctitle = "Alphabetically sorted index"
}
%% http://lsr.di.unimi.it/LSR/Item?id=763

% Usage:
% - define and index item with \indexItem $sortstring $markup
% - use \indexSection $sortstring $markup to divide the index into several sections
% - display the alphabetical index with \markuplist \index

% code ist mostly taken from ./ly/toc-init.ly and just renamed and slightly modfied

%here starts the snippet:

%% defined later, in a closure
#(define-public (add-index-item! markup-symbol text sorttext) #f)
#(define-public (index-items) #f)

#(let ((index-item-list (list)))
   (set! add-index-item!
   (lambda (markup-symbol text sorttext)
     (let ((label (gensym "index")))
       (set! index-item-list
       ;; We insert index items sorted from the beginning on and do
       ;; not sort them later - this saves pretty much computing time
       (insert-alphabetical-sorted! (list label markup-symbol text sorttext)
       index-item-list))
       (make-music 'EventChord
         'page-marker #t
         'page-label label
         'elements (list (make-music 'LabelEvent
         'page-label label))))))
   (set! index-items (lambda ()
         index-item-list)))

#(define (insert-alphabetical-sorted! iitem ilist)
  (if
    (null? ilist) (list iitem)
    (if
      (string-ci<? (cadddr iitem) (cadddr (car ilist))) (cons iitem ilist)
      (cons (car ilist) (insert-alphabetical-sorted! iitem (cdr ilist)))
    )
  )
)

\header { tagline = ##f }

\paper {
  indexTitleMarkup = \markup \column {
    \fontsize #5 \sans \bold \fill-line { \null "Alphabetical Index" \null }
    \hspace #1
  }
  indexItemMarkup = \markup \large \fill-line {
    \fromproperty #'index:text
    \fromproperty #'index:page
  }
  indexSectionMarkup = \markup \column {
    \hspace #1
    \fill-line { \sans \bold \fontsize #3 \fromproperty #'index:text }
    \hspace #1
  }
}

#(define-markup-list-command (index layout props) ()
  ( _i "Outputs an alphabetical sorted index, using the paper
  variable @code{indexTitleMarkup} for its title, then the list of
  lines built using the @code{indexItem} music function
  Usage: @code{\\markuplist \\index}" )
  (cons (interpret-markup layout props
        (ly:output-def-lookup layout 'indexTitleMarkup))
  (space-lines (chain-assoc-get 'baseline-skip props)
        (map (lambda (index-item)
         (let ((label (car index-item))
         (index-markup (cadr index-item))
         (text (caddr index-item)))
           (interpret-markup
             layout
             (cons (list (cons 'index:page
              (markup #:page-ref label "XXX" "?"))
             (cons 'index:text text))
             props)
             (ly:output-def-lookup layout index-markup))))
       (index-items)))))

indexItem =
#(define-music-function (sorttext text) (string? markup?)
   "Add a line to the alphabetical index, using the @code{indexItemMarkup} paper
variable markup."
   (add-index-item! 'indexItemMarkup text sorttext))

indexSection =
#(define-music-function (sorttext text) (string? markup?)
   "Add a section line to the alphabetical index, using @code{indexSectionMarkup}
paper variable markup. This can be used to divide the alphabetical index into
different sections, for example one section for each first letter."
   (add-index-item! 'indexSectionMarkup text sorttext))


% ---------------------------------------------------------------
% Example

\markuplist \index

\indexSection #"B" \markup { "B" }
\indexSection #"K" \markup { "K" }
\indexSection #"Z" \markup { "Z" }


  \score {
    \relative c' {
      \repeat unfold 10 { c d e f }
      \indexItem #"Karola" \markup{ "Karola" }
      \repeat unfold 10 { c d e f }
      \bar "|."
    }
    \header {
      piece = "first piece"
    }
  }
  \score {
    <<
      \new Staff \new Voice = "voc" {
        \relative c' {
          \indexItem #"Zora" \markup { \italic "Zora" }
          \repeat unfold 10 { c d e f }
          \indexItem #"Bettina" \markup { \larger "Bettina" }
          \indexItem #"Barbara" \markup { \smallCaps "Barbara" }
          \repeat unfold 10 { c d e f }
          \bar "|."
        }
      }
      \new Lyrics \lyricsto "voc" {
        \repeat unfold 10 { la la la la }
        \repeat unfold 10 { mi mi mi mi }
    } >>
    \header {
      piece = "second piece"
    }
  }

